/*
 * PublishHandler.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1999-2002 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, suhler.
 *
 * Version:  2.1
 * Created by suhler on 99/03/31
 * Last modified by suhler on 02/10/01 16:36:27
 */

package sunlabs.brazil.handler;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Properties;
import java.util.StringTokenizer;
import sunlabs.brazil.server.FileHandler;
import sunlabs.brazil.server.Handler;
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;

/**
 * Handler for supporting publishing from Communicator.
 * Launches an authentication handler to protect the content
 * from malicious users.
 * <p>
 * Looks for <code>PUT</code> requests, and creates or modifies the
 * content as indicated.
 * <p>
 * The following request properties are used:
 * <dl class=props>
 * <dt>prefix, suffix, glob, match
 * <dd>Specify the URL that triggers this handler.
 * (See {@link MatchString}).
 * <dt>session	<dd> The the name of request property holding the session
 *		information to provide the credentials for posting.  The
 *		default is "SessionID".
 * </dl>
 *
 * @author		Stephen Uhler
 * @version		2.1, 02/10/01
 */

public class PublishHandler implements Handler {
    private static final String SESSION = "session";

    MatchString isMine;            // check for matching url
    public String session = "SessionID";

    public String propsPrefix;    // my prefix into the properties file

    /**
     * Start up the authentication handler.
     */

    public boolean
    init(Server server, String prefix) {
	this.propsPrefix = prefix;

	Properties props = server.props;
	isMine = new MatchString(prefix, server.props);
	session = props.getProperty(prefix + SESSION, session);

	return true;
    }

    /**
     * Make sure this is one of our "PUT" requests.
     * Look up the credentials for this request.
     * If no credentials are found, prompt the user for them.
     * IF OK, save file to proper spot.
     */

    public boolean
    respond(Request request)
	throws IOException
    {
    	if (!isMine.match(request.url) || !request.method.equals("PUT")) {
	    return false;
	}	

	/*
	 * screen out bad requests
	 */

	if (request.postData == null) {
	    request.sendError(400, "No content to put");
	    return true;
	}

	if (request.headers.get("Content-Range") != null) {
	    request.sendError(501, "Can't handle partial puts");
	    return true;
	}

	/*
	 * Get the credentials left by the auth handler.
	 */

	String credentials = request.props.getProperty(session, "");

	/*
 	 * Make sure the credentials are valid for this prefix
	 */

	StringTokenizer st = new StringTokenizer(credentials);
	boolean ok = false;
	while (st.hasMoreTokens()) {
	    String prefix = st.nextToken();
	    if (request.url.startsWith(prefix)) {
		ok = true;
		request.log(Server.LOG_DIAGNOSTIC, propsPrefix,
			"file: " + request.url + " matches: " + prefix);
		break;
	    }
	}

    	/*
    	 * OK, now try to save the file.
    	 */

	String root = request.props.getProperty(FileHandler.ROOT, ".");
	File file = new File(root + FileHandler.urlToPath(request.url));

	request.log(Server.LOG_INFORMATIONAL, propsPrefix + "root: " + root);

	int code = (file.exists()) ? 204 : 201;

	FileOutputStream out;
	try {
	    out = new FileOutputStream(file);
	} catch (IOException e) {
	    request.sendError(403, "Permission denied", file.getAbsolutePath());
	    return true;
	}
	out.write(request.postData);
	out.close();
	request.sendResponse("Update of " + request.url + " succeeded",
		"text/html", code);
	return true;
    }
}
